<?php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");

require_once __DIR__ . '/auth.php';
require_once("db.php");

$userId = authenticate(); // سيعيد user_id الصحيح

$data = json_decode(file_get_contents("php://input"), true);

if (!isset($data['product_id'], $data['action'])) {
    echo json_encode(["status" => "error", "message" => "بيانات ناقصة"]);
    exit;
}

$productId = intval($data['product_id']);
$action    = $data['action'];

// التحقق من صلاحية user_id لمنع الزائرين من الإعجاب
if ($userId <= 0) {
    echo json_encode(["status" => "error", "message" => "يرجى تسجيل الدخول للإعجاب بالمنتج"]);
    exit;
}

if (!in_array($action, ['like', 'unlike'])) {
    echo json_encode(["status" => "error", "message" => "إجراء غير صالح"]);
    exit;
}

// تأكد أن المنتج موجود
$checkProduct = $conn->prepare("SELECT id FROM products WHERE id = ?");
$checkProduct->bind_param("i", $productId);
$checkProduct->execute();
$checkProduct->store_result();
if ($checkProduct->num_rows === 0) {
    echo json_encode(["status" => "error", "message" => "المنتج غير موجود"]);
    exit;
}
$checkProduct->close();

$conn->begin_transaction();

try {
    if ($action === 'like') {
        $stmt = $conn->prepare("
            INSERT INTO product_likes (product_id, user_id)
            VALUES (?, ?)
            ON DUPLICATE KEY UPDATE product_id = product_id
        ");
        $stmt->bind_param("ii", $productId, $userId);
        $stmt->execute();

        if ($conn->affected_rows === 1) {
            $inc = $conn->prepare("UPDATE products SET likes_count = likes_count + 1 WHERE id = ?");
            $inc->bind_param("i", $productId);
            $inc->execute();
            $inc->close();
            $status = "success";
            $message = "Liked";
        } else {
            $status = "already_liked";
            $message = "You already liked this product";
        }
        $stmt->close();

    } else { // unlike
        $stmt = $conn->prepare("DELETE FROM product_likes WHERE product_id = ? AND user_id = ?");
        $stmt->bind_param("ii", $productId, $userId);
        $stmt->execute();

        if ($stmt->affected_rows > 0) {
            $dec = $conn->prepare("UPDATE products SET likes_count = GREATEST(likes_count - 1, 0) WHERE id = ?");
            $dec->bind_param("i", $productId);
            $dec->execute();
            $dec->close();

            $status = "success";
            $message = "Unliked";
        } else {
            $status = "not_liked";
            $message = "You haven't liked this product";
        }
        $stmt->close();
    }

    // اجلب القيمة الحالية لعدد اللايكات
    $result = $conn->prepare("SELECT likes_count FROM products WHERE id = ?");
    $result->bind_param("i", $productId);
    $result->execute();
    $result->bind_result($likesCount);
    $result->fetch();
    $result->close();
    
    $likesCount = $likesCount ?? 0;

    $conn->commit();

    echo json_encode([
        "status" => $status,
        "message" => $message,
        "likes_count" => intval($likesCount),
    ]);
} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(["status" => "error", "message" => "Database error"]);
}

$conn->close();
?>
