// ===============================
// Modern Table Functionality
// ===============================

document.addEventListener('DOMContentLoaded', function() {
    initTableSearch();
    initTableSorting();
    initTableFilters();
    initPaymentCalculators();
    initAnimations();
});

// ===============================
// Table Search
// ===============================
function initTableSearch() {
    const searchInputs = document.querySelectorAll('#tableSearch, #searchInput, .search-box input');
    
    searchInputs.forEach(input => {
        if (!input) return;
        
        input.addEventListener('input', function() {
            const searchValue = this.value.toLowerCase().trim();
            const table = this.closest('.table-wrapper')?.querySelector('table') || 
                         document.querySelector('table');
            
            if (!table) return;
            
            const rows = table.querySelectorAll('tbody tr');
            let visibleCount = 0;
            
            rows.forEach((row) => {
                const rowText = row.textContent.toLowerCase();
                const matches = rowText.includes(searchValue);
                
                row.style.display = matches ? '' : 'none';
                if (matches) visibleCount++;
            });
            
            showNoResultsMessage(table, visibleCount === 0 && searchValue.length > 0);
        });
    });
}

// ===============================
// Table Sorting
// ===============================
function initTableSorting() {
    const tables = document.querySelectorAll('table');
    
    tables.forEach(table => {
        const headers = table.querySelectorAll('th');
        
        headers.forEach((header, index) => {
            if (header.classList.contains('no-sort')) return;
            
            header.classList.add('sortable');
            header.style.cursor = 'pointer';
            
            header.addEventListener('click', function() {
                sortTable(table, index, this);
            });
        });
    });
}

function sortTable(table, columnIndex, header) {
    const tbody = table.querySelector('tbody');
    if (!tbody) return;
    
    const rows = Array.from(tbody.querySelectorAll('tr'));
    const isAscending = header.classList.contains('sort-asc');
    
    table.querySelectorAll('th').forEach(th => {
        th.classList.remove('sort-asc', 'sort-desc');
    });
    
    header.classList.remove('sort-asc', 'sort-desc');
    header.classList.add(isAscending ? 'sort-desc' : 'sort-asc');
    
    rows.sort((a, b) => {
        const aText = a.cells[columnIndex]?.textContent.trim() || '';
        const bText = b.cells[columnIndex]?.textContent.trim() || '';
        
        const aNum = parseFloat(aText.replace(/[^0-9.-]/g, ''));
        const bNum = parseFloat(bText.replace(/[^0-9.-]/g, ''));
        
        if (!isNaN(aNum) && !isNaN(bNum)) {
            return isAscending ? bNum - aNum : aNum - bNum;
        }
        
        return isAscending 
            ? bText.localeCompare(aText, 'ar')
            : aText.localeCompare(bText, 'ar');
    });
    
    rows.forEach(row => tbody.appendChild(row));
}

// ===============================
// Table Filters
// ===============================
function initTableFilters() {
    const filterSelects = document.querySelectorAll('.filter-select');
    
    filterSelects.forEach(select => {
        select.addEventListener('change', function() {
            const filterValue = this.value.toLowerCase();
            const filterColumn = this.dataset.column || this.getAttribute('data-column');
            const table = this.closest('.table-wrapper')?.querySelector('table') || 
                         document.querySelector('table');
            
            if (!table || !filterColumn) return;
            
            const headers = Array.from(table.querySelectorAll('th'));
            const columnIndex = headers.findIndex(th => {
                const text = th.textContent.trim();
                return text.includes(filterColumn);
            });
            
            if (columnIndex === -1) return;
            
            const rows = table.querySelectorAll('tbody tr');
            
            rows.forEach(row => {
                if (filterValue === 'all' || filterValue === '') {
                    row.style.display = '';
                    return;
                }
                
                const cellText = row.cells[columnIndex]?.textContent.toLowerCase() || '';
                row.style.display = cellText.includes(filterValue) ? '' : 'none';
            });
        });
    });
}

// ===============================
// Payment Calculators
// ===============================
function initPaymentCalculators() {
    const paymentForms = document.querySelectorAll('.payment-form');
    
    paymentForms.forEach(form => {
        const tierSelect = form.querySelector('.new_tier, select[name="new_tier"]');
        const typeSelect = form.querySelector('.subscription_type, select[name="subscription_type"]');
        const amountSpan = form.querySelector('.calculated_amount');
        
        if (!tierSelect || !typeSelect || !amountSpan) return;
        
        const tiers = window.tiers || {};
        const discounts = window.discounts || {};
        
        function calculateAmount() {
            const tier = tierSelect.value;
            const subscription = typeSelect.value;
            
            let base = tiers[tier] || 0;
            let multiplier = subscription === 'semi-annual' ? 6 : 
                           subscription === 'yearly' ? 12 : 1;
            
            let total = base * multiplier;
            let discountPercent = discounts[subscription] || 0;
            let finalAmount = total * (1 - discountPercent / 100);
            
            amountSpan.textContent = finalAmount.toFixed(2);
            
            amountSpan.style.transition = 'transform 0.2s';
            amountSpan.style.transform = 'scale(1.1)';
            setTimeout(() => {
                amountSpan.style.transform = 'scale(1)';
            }, 200);
        }
        
        calculateAmount();
        tierSelect.addEventListener('change', calculateAmount);
        typeSelect.addEventListener('change', calculateAmount);
    });
}

// ===============================
// Animations & UX Enhancements
// ===============================
function initAnimations() {
    const tables = document.querySelectorAll('table');
    tables.forEach((table, index) => {
        table.style.opacity = '0';
        setTimeout(() => {
            table.style.transition = 'opacity 0.5s ease-in';
            table.style.opacity = '1';
        }, index * 100);
    });
    
    document.querySelectorAll('.btn').forEach(btn => {
        btn.addEventListener('click', function(e) {
            const ripple = document.createElement('span');
            const rect = this.getBoundingClientRect();
            const size = Math.max(rect.width, rect.height);
            const x = e.clientX - rect.left - size / 2;
            const y = e.clientY - rect.top - size / 2;
            
            ripple.style.cssText = `
                position: absolute;
                width: ${size}px;
                height: ${size}px;
                left: ${x}px;
                top: ${y}px;
                border-radius: 50%;
                background: rgba(255, 255, 255, 0.6);
                transform: scale(0);
                animation: ripple-animation 0.6s ease-out;
                pointer-events: none;
            `;
            
            this.appendChild(ripple);
            setTimeout(() => ripple.remove(), 600);
        });
    });
}

// ===============================
// No Results Message
// ===============================
function showNoResultsMessage(table, show) {
    let message = table.parentElement.querySelector('.no-results');
    
    if (show && !message) {
        message = document.createElement('div');
        message.className = 'no-results alert alert-info';
        message.textContent = 'لا توجد نتائج مطابقة للبحث';
        message.style.textAlign = 'center';
        message.style.marginTop = '1rem';
        table.parentElement.appendChild(message);
    } else if (!show && message) {
        message.remove();
    }
}

// Add ripple animation CSS
if (!document.getElementById('ripple-style')) {
    const style = document.createElement('style');
    style.id = 'ripple-style';
    style.textContent = `
        .btn {
            position: relative;
            overflow: hidden;
        }
        @keyframes ripple-animation {
            to {
                transform: scale(4);
                opacity: 0;
            }
        }
    `;
    document.head.appendChild(style);
}